/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Data;
using System.Data.SqlTypes;
using System.Globalization;

using Borland.Eco.Persistence;
using Borland.Eco.Persistence.Connection;
using Borland.Eco.Persistence.Default;
using Borland.Eco.Persistence.Configuration;

namespace Borland.Eco.Persistence.Sybase
{
	public sealed class SingleAsFloat: AbstractNumericSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "FLOAT"; // do not localize
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Single));
			parameter.DbType = DbType.Double;
			if (value == null)
				parameter.Value = DBNull.Value;
			else
				parameter.Value = (System.Double)(System.Single)value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToSingle(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Double));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Single)(System.Double)columnValue;
		}
	}

	///<summary>
	///Mapper between an AutoInc value of type Int32 and its persistent representation as NUMERIC(9,0).
	///</summary>
	public sealed class Int32AsNumericIdentity: AbstractNumericSingleColumnAttribute, IAutoIncAttributemapping
	{
		public override String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
				return "";
		}

		public String ColumnType(int length)
		{
			return "NUMERIC(9,0) IDENTITY"; // do not localize
		}

		///<summary>This method should never be called. Autoinc values are never stored!</summary>
		///<exception cref="NotSupportedException">Thrown always.</exception>
		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			throw new NotSupportedException(PersistenceStringRes.sDoNotCallInt32AsAutoIncValueToParameter);
		}

		///<summary>This method should never be called. Autoinc values are never stored!</summary>
		///<exception cref="NotSupportedException">Thrown always.</exception>
		public void StringToParameter(string value, IDataParameter parameter)
		{
			throw new NotSupportedException(PersistenceStringRes.sDoNotCallInt32AsAutoIncStringToParameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Decimal));
			if (DBNull.Value.Equals(columnValue))
				return null;
			else
				return (System.Int32)(System.Decimal)columnValue;
		}

		public void MakeColumnAutoInc(string tableName, string columnName, IDatabase database)
		{
			// Intentionally left blank.
		}
	}

	public sealed class BooleanAsBit: AbstractSingleColumnAttribute, ISingleColumnAttributemapping
	{
		public String ColumnType(int length)
		{
			return "BIT"; // do not localize
		}

		public String DefaultDbValue(string modeledValue, SqlDatabaseConfig sqlDatabaseConfig)
		{
			if ((modeledValue == null) || (modeledValue.Length == 0))
				return 0.ToString(CultureInfo.InvariantCulture); // do not localize
			else
				return modeledValue;
		}

		public void ValueToParameter(System.Object value, IDataParameter parameter)
		{
			if (parameter == null)
				throw new ArgumentNullException("parameter");
			EnsureType(value, typeof(System.Boolean));
			parameter.DbType = DbType.Boolean ;
			if (value == null)
				parameter.Value = DBNull.Value;
			else 
				parameter.Value = value;
		}

		public void StringToParameter(string value, IDataParameter parameter)
		{
			ValueToParameter(MappingUtils.ConvertStringToBoolean(value), parameter);
		}

		public override System.Object ColumnToValue(System.Object columnValue)
		{
			EnsureType(columnValue, typeof(System.Boolean));
			if (columnValue == DBNull.Value)
				return null;
			else 
				return (System.Boolean)columnValue;
		}
	}
}

